"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.wsConnectionHandler = void 0;
const node_fs_1 = __importDefault(require("node:fs"));
const axios_1 = __importDefault(require("axios"));
const logging_1 = require("../logging");
const constants_1 = require("../constants");
const getProLinuxInfo_1 = require("../helpers/getProLinuxInfo");
const systemStateContainer_1 = require("../state/systemStateContainer");
const secureSwitchRole_1 = require("../modules/pl2/server/secureSwitchRole");
const wsConnectionHandler = (socket, config, cloud, localSocketBroadcast) => {
    logging_1.log.info("Client connected to ProLinuxD!");
    const reply = (msg) => {
        socket.send(JSON.stringify(msg));
    };
    socket.on("message", async (data) => {
        try {
            let msg = JSON.parse(data.toString());
            const replyResult = (forAction, status, data) => {
                reply({
                    action: constants_1.LocalActions.RESULT,
                    payload: {
                        forAction: forAction,
                        status: status,
                        data: data
                    },
                    id: msg.id ?? null
                });
            };
            logging_1.log.info("[Local] Received message: " + JSON.stringify(msg));
            switch (msg.action) {
                case constants_1.LocalActions.LOG:
                    {
                        (0, logging_1.logger)(msg.payload.msg, msg.payload.type, msg.payload.from);
                    }
                    break;
                case constants_1.LocalActions.DEVICE_STREAM_TERMINAL:
                    {
                        cloud?.callWS("device-stream-terminal", {
                            deviceUUID: cloud.uuid,
                            fromDevice: true,
                            text: msg.payload.data
                        }, false);
                    }
                    break;
                case constants_1.LocalActions.SET_TOKEN:
                    {
                        config.ocs2.access_token = msg.payload.token;
                        replyResult(constants_1.LocalActions.SET_TOKEN, true, {});
                    }
                    break;
                case constants_1.LocalActions.SET_SELECTED_ROOT:
                    {
                        config.pl2.selected_root = msg.payload.selectedRoot;
                        replyResult(constants_1.LocalActions.SET_SELECTED_ROOT, true, {});
                    }
                    break;
                case constants_1.LocalActions.SET_LOCKED_ROOT:
                    {
                        config.pl2.locked_root = msg.payload.lockedRoot;
                        replyResult(constants_1.LocalActions.SET_LOCKED_ROOT, true, {});
                    }
                    break;
                case constants_1.LocalActions.SET_HOSTNAME:
                    {
                        config.pl2.hostname = msg.payload.hostname;
                        replyResult(constants_1.LocalActions.SET_HOSTNAME, true, {});
                    }
                    break;
                case constants_1.LocalActions.SET_DISABLE_KEXEC:
                    {
                        config.pl2.disable_kexec = msg.payload.disableKexec;
                        replyResult(constants_1.LocalActions.SET_DISABLE_KEXEC, true, {});
                    }
                    break;
                case constants_1.LocalActions.SET_REMOTE_API:
                    {
                        config.pl2.remote_api = msg.payload.remoteAPI;
                        replyResult(constants_1.LocalActions.SET_REMOTE_API, true, {});
                    }
                    break;
                case constants_1.LocalActions.STATUS:
                    {
                        logging_1.log.info("Sending status...");
                        socket.send(JSON.stringify({
                            action: constants_1.LocalActions.RESULT,
                            payload: {
                                forAction: constants_1.LocalActions.STATUS,
                                status: true,
                                data: {
                                    status: "ok",
                                    ocsConnnected: cloud?.connected ?? false,
                                    ocsReady: cloud?.ready ?? false,
                                    modules: config.prolinuxd.modules,
                                    selectedRoot: config.pl2.selected_root,
                                    lockedRoot: config.pl2.locked_root,
                                    hostname: config.pl2.hostname,
                                    disableKexec: config.pl2.disable_kexec,
                                    buildInfo: await (0, getProLinuxInfo_1.getProLinuxInfo)(),
                                    emptyPersistRoot: node_fs_1.default.readdirSync("/sineware/persistroot").length == 0,
                                    config: config
                                },
                            },
                            id: msg.id ?? null
                        }));
                    }
                    break;
                case constants_1.LocalActions.GET_LOGS:
                    {
                        socket.send(JSON.stringify({
                            action: constants_1.LocalActions.RESULT,
                            payload: {
                                forAction: constants_1.LocalActions.GET_LOGS,
                                status: true,
                                data: {
                                    logs: logging_1.log.getLogs()
                                }
                            },
                            id: msg.id ?? null
                        }));
                    }
                    break;
                case constants_1.LocalActions.GET_UPDATE_INFO:
                    {
                        const info = await (0, getProLinuxInfo_1.getProLinuxInfo)();
                        let res = await axios_1.default.get(`https://update.sineware.ca/updates/${info.product}/${info.variant}/${info.channel}`);
                        socket.send(JSON.stringify({
                            action: constants_1.LocalActions.RESULT,
                            payload: {
                                forAction: constants_1.LocalActions.GET_UPDATE_INFO,
                                status: true,
                                data: {
                                    update: res.data,
                                    updateAvailable: (res.data.buildnum > info.buildnum)
                                }
                            },
                            id: msg.id ?? null
                        }));
                    }
                    break;
                case constants_1.LocalActions.START_UPDATE:
                    {
                        try {
                            // Check if /sineware/persistroot is empty. If it's not, fail.
                            if (node_fs_1.default.readdirSync("/sineware/persistroot").length > 0) {
                                replyResult(constants_1.LocalActions.START_UPDATE, false, {
                                    msg: "You have persistent changes in your rootfs! Please disable the root-lock and reset-writable."
                                });
                                return;
                            }
                            const info = await (0, getProLinuxInfo_1.getProLinuxInfo)();
                            let updateInfo = await axios_1.default.get(`https://update.sineware.ca/updates/${info.product}/${info.variant}/${info.channel}`);
                            const newRoot = (config.pl2.selected_root === "a") ? "b" : "a";
                            // Download the update from http://cdn.sineware.ca/repo/${info.product}/${info.variant}/${info.channel}/${res.data.arch}/${info.filename}.squish to /sineware/prolinux_${newRoot}.squish
                            // and send update-progress events
                            const { data, headers } = await (0, axios_1.default)({
                                method: 'get',
                                url: `http://cdn.sineware.ca/repo/${info.product}/${info.variant}/${info.channel}/${info.arch}/${info.filename}`,
                                responseType: 'stream'
                            });
                            const totalLength = headers['content-length'];
                            const writer = node_fs_1.default.createWriteStream(`/sineware/prolinux_${newRoot}.squish`);
                            let progress = 0;
                            data.on('data', (chunk) => {
                                localSocketBroadcast({
                                    action: constants_1.LocalActions.UPDATE_PROGRESS,
                                    payload: {
                                        progress: progress += chunk.length,
                                        total: totalLength,
                                        newRoot: newRoot,
                                        buildnum: updateInfo.data.buildnum,
                                    }
                                });
                                if (progress == totalLength) {
                                    config.pl2.selected_root = newRoot;
                                }
                            });
                            data.pipe(writer);
                            replyResult(constants_1.LocalActions.START_UPDATE, true, {});
                        }
                        catch (e) {
                            replyResult(constants_1.LocalActions.START_UPDATE, false, {
                                msg: e.message
                            });
                        }
                    }
                    break;
                case constants_1.LocalActions.DESCRIBE_API:
                    {
                        replyResult(constants_1.LocalActions.DESCRIBE_API, true, {
                            actions: constants_1.LocalActions
                        });
                    }
                    break;
                case constants_1.LocalActions.SET_RESET_PERSISTROOT_FLAG:
                    {
                        node_fs_1.default.writeFileSync("/sineware/data/.reset_persistroot", "1");
                        replyResult(constants_1.LocalActions.SET_RESET_PERSISTROOT_FLAG, true, {});
                    }
                    break;
                case constants_1.LocalActions.RUNTIME_VERIFY_STATE_INTEGRITY:
                    {
                        const res = (0, systemStateContainer_1.verifyStateIntegrity)();
                        replyResult(constants_1.LocalActions.RUNTIME_VERIFY_STATE_INTEGRITY, res.valid, res);
                    }
                    break;
                // Server actions
                case constants_1.LocalActions.SERVER_STATUS:
                    {
                        replyResult(constants_1.LocalActions.SERVER_STATUS, true, {
                            status: "ok",
                            config: systemStateContainer_1.state.extraConfig.server_roles,
                            roles: constants_1.ServerRoleType
                        });
                    }
                    break;
                case constants_1.LocalActions.SERVER_ROLE_ENABLE:
                    {
                        const role = msg.payload.role;
                        if (role == constants_1.ServerRoleType.WEBSERVER) {
                            systemStateContainer_1.state.extraConfig.server_roles.webserver.enabled = true;
                        }
                        else if (role == constants_1.ServerRoleType.SECURE_SWITCH_APPLIANCE) {
                            let res = await (0, secureSwitchRole_1.setupSecureSwitchRole)();
                            if (res) {
                                systemStateContainer_1.state.extraConfig.server_roles.secure_switch.enabled = true;
                                replyResult(constants_1.LocalActions.SERVER_ROLE_ENABLE, true, {});
                            }
                            else {
                                replyResult(constants_1.LocalActions.SERVER_ROLE_ENABLE, false, {
                                    msg: "Failed to start SecureSwitch Appliance Server Role"
                                });
                            }
                        }
                    }
                    break;
                case constants_1.LocalActions.SERVER_ROLE_DISABLE:
                    {
                        const role = msg.payload.role;
                        if (role == constants_1.ServerRoleType.WEBSERVER) {
                            systemStateContainer_1.state.extraConfig.server_roles.webserver.enabled = false;
                        }
                        else if (role == constants_1.ServerRoleType.SECURE_SWITCH_APPLIANCE) {
                            await (0, secureSwitchRole_1.deleteSecureSwitchRole)();
                            systemStateContainer_1.state.extraConfig.server_roles.secure_switch.enabled = false;
                        }
                        else {
                            replyResult(constants_1.LocalActions.SERVER_ROLE_DISABLE, false, {
                                msg: "Invalid role, must be one of: " + Object.values(constants_1.ServerRoleType).join(", ")
                            });
                        }
                    }
                    break;
            }
        }
        catch (e) {
            console.log(e);
            reply({
                action: constants_1.LocalActions.ERROR,
                payload: {
                    msg: e.message
                }
            });
        }
    });
    socket.on("close", () => {
        logging_1.log.info("Client disconnected from ProLinuxD!");
    });
};
exports.wsConnectionHandler = wsConnectionHandler;
//# sourceMappingURL=wsConnectionHandler.js.map